/**
 * Rezume Popup Script
 * Handles all popup UI logic:
 *   - Resume upload + login flow
 *   - JD extraction from current tab
 *   - Sending resume + JD to /api/match
 *   - Rendering match results
 */

'use strict';

// ---------------------------------------------------------------------------
// Config — update API_BASE for production deployment
// ---------------------------------------------------------------------------
// const API_BASE = 'http://localhost:3000'; // --- LOCAL DEV ---
const API_BASE = 'https://rezume.work'; // --- PRODUCTION ---


// ---------------------------------------------------------------------------
// SWE tech keywords — mirrored from content.js for client-side quick match.
// Used to compare what keywords appear in both the JD and the resume.
// ---------------------------------------------------------------------------
const SWE_TECH_KEYWORDS = [
  // Languages
  'python', 'javascript', 'typescript', 'java', 'c++', 'golang', 'go', 'rust',
  'ruby', 'php', 'swift', 'kotlin', 'scala', 'sql', 'html', 'css', 'bash', 'shell',
  'r language', 'matlab', 'haskell', 'elixir', 'clojure', 'groovy',
  // Frameworks & libraries
  'react', 'angular', 'vue', 'next.js', 'nuxt', 'svelte', 'node.js', 'express',
  'django', 'flask', 'fastapi', 'spring boot', 'spring', '.net', 'asp.net',
  'rails', 'laravel', 'symfony', 'tensorflow', 'pytorch', 'pandas', 'numpy',
  'scikit-learn', 'keras', 'jquery', 'tailwind',
  // Databases
  'postgresql', 'postgres', 'mysql', 'mongodb', 'redis', 'cassandra', 'dynamodb',
  'elasticsearch', 'oracle', 'sqlite', 'neo4j', 'couchdb', 'influxdb',
  'mariadb', 'sql server', 'snowflake', 'bigquery', 'redshift',
  // Cloud & infrastructure
  'aws', 'azure', 'gcp', 'google cloud', 'ec2', 's3', 'lambda', 'rds',
  'cloudformation', 'ecs', 'eks', 'cloudfront', 'vpc', 'iam',
  'heroku', 'digitalocean', 'vercel', 'netlify',
  // DevOps & tooling
  'docker', 'kubernetes', 'k8s', 'terraform', 'ansible', 'jenkins',
  'github actions', 'gitlab ci', 'circleci', 'travis ci', 'helm',
  'prometheus', 'grafana', 'datadog', 'new relic', 'splunk',
  'nginx', 'apache', 'git', 'github', 'gitlab', 'bitbucket',
  // Architecture & patterns
  'microservices', 'rest api', 'restful', 'graphql', 'grpc', 'kafka', 'rabbitmq',
  'ci/cd', 'serverless', 'api gateway', 'load balancing', 'message queue',
  'event-driven', 'service mesh', 'istio', 'caching', 'cdn',
  // Methodologies & practices
  'agile', 'scrum', 'kanban', 'devops', 'sre', 'tdd', 'bdd',
  'pair programming', 'continuous integration', 'continuous deployment', 'gitops',
  // Testing
  'jest', 'mocha', 'pytest', 'junit', 'selenium', 'cypress',
  'unit testing', 'integration testing', 'end-to-end testing',
  // Role titles that appear in JD bodies
  'software engineer', 'backend engineer', 'frontend engineer', 'full stack engineer',
  'devops engineer', 'data engineer', 'ml engineer', 'mobile engineer',
  'platform engineer', 'infrastructure engineer', 'site reliability engineer',
];

// Configure PDF.js worker to use the locally bundled file (MV3 requires this)
if (typeof pdfjsLib !== 'undefined') {
  pdfjsLib.GlobalWorkerOptions.workerSrc = chrome.runtime.getURL('vendor/pdf.worker.min.js');
}

// ---------------------------------------------------------------------------
// DOM refs
// ---------------------------------------------------------------------------
const viewLogin         = document.getElementById('viewLogin');
const viewMain          = document.getElementById('viewMain');
const userChip          = document.getElementById('userChip');
const userAvatar        = document.getElementById('userAvatar');
const userNameEl        = document.getElementById('userName');
const userEmailEl       = document.getElementById('userEmail');
const btnLogout         = document.getElementById('btnLogout');
const btnChangeResume   = document.getElementById('btnChangeResume');

// Login view
const dropZone          = document.getElementById('dropZone');
const resumeFileInput   = document.getElementById('resumeFileInput');
const loginFileName     = document.getElementById('loginFileName');
const btnLogin          = document.getElementById('btnLogin');

// Main view
const jdMeta            = document.getElementById('jdMeta');
const jdPreview         = document.getElementById('jdPreview');

// Results - new floating card structure
const scoreCard         = document.getElementById('scoreCard');
const keywordsCard      = document.getElementById('keywordsCard');
const tailorCTACard     = document.getElementById('tailorCTACard');
const tailorCard        = document.getElementById('tailorCard');
const scoreRingFill     = document.getElementById('scoreRingFill');
const scoreValue        = document.getElementById('scoreValue');
const scoreLevel        = document.getElementById('scoreLevel');
const scoreMeta         = document.getElementById('scoreMeta');
const scoreVerdict      = document.getElementById('scoreVerdict');
const criticalGroup     = document.getElementById('criticalGroup');
const criticalPills     = document.getElementById('criticalPills');
const recommendedGroup  = document.getElementById('recommendedGroup');
const recommendedPills  = document.getElementById('recommendedPills');
const foundGroup        = document.getElementById('foundGroup');
const foundPills        = document.getElementById('foundPills');

// Status
const statusBar         = document.getElementById('statusBar');
const spinner           = document.getElementById('spinner');
const statusText        = document.getElementById('statusText');

// Tailor
const btnTailor         = document.getElementById('btnTailor');
const tailorProgress    = document.getElementById('tailorProgress');
const tailorProgressText = document.getElementById('tailorProgressText');
const tailorDone        = document.getElementById('tailorDone');
const tailorDoneLabel   = document.getElementById('tailorDoneLabel');
const btnDownloadTailor = document.getElementById('btnDownloadTailor');

// ---------------------------------------------------------------------------
// State
// ---------------------------------------------------------------------------
let selectedLoginFile   = null;   // File chosen for login
let extractedJD         = null;   // { text, url, title, method, confidence }
let cachedToken         = null;   // JWT stored in extension storage
let cachedUser          = null;   // { name, email }
let cachedResumeBase64  = null;   // Stored resume (base64)
let cachedResumeName    = null;   // Stored resume filename
let cachedResumeKeywords = null;  // Keywords extracted from the stored resume (array)
let tailoredPdfBlob     = null;   // Last tailored PDF blob (for download)
let tailoredText        = null;   // Last tailored text (for regenerating PDF)
let lastMatchResults    = null;   // Cached match results for JD tailoring
let previousMatchScore  = null;   // Previous match score (for celebration animation)

// Cache match results per job URL to avoid re-running analysis
// Key: job URL, Value: { matchResults, extractedText, matchData, analysisData, tailoredPdfBlob, tailoredFilename, previousScore, timestamp }
let matchCache = {};

// ---------------------------------------------------------------------------
// Utility helpers
// ---------------------------------------------------------------------------

function bg(action, payload = {}) {
  return new Promise((resolve) =>
    chrome.runtime.sendMessage({ action, ...payload }, resolve)
  );
}

function show(el)  { el.classList.remove('hidden'); }
function hide(el)  { el.classList.add('hidden'); }

function setStatus(msg, type = 'loading') {
  show(statusBar);
  statusBar.className = `status-bar ${type}`;
  statusText.textContent = msg;
  if (type === 'loading') show(spinner);
  else hide(spinner);
}

function clearStatus() {
  hide(statusBar);
  hide(spinner);
}

function fileToBase64(file) {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onload  = () => resolve(reader.result.split(',')[1]); // strip data:... prefix
    reader.onerror = reject;
    reader.readAsDataURL(file);
  });
}

function base64ToBlob(b64, type = 'application/pdf') {
  const binary = atob(b64);
  const bytes  = new Uint8Array(binary.length);
  for (let i = 0; i < binary.length; i++) bytes[i] = binary.charCodeAt(i);
  return new Blob([bytes], { type });
}

function initials(name) {
  if (!name) return '?';
  return name.split(' ').slice(0, 2).map(w => w[0]).join('').toUpperCase();
}

// ---------------------------------------------------------------------------
// IndexedDB for persistent tailored PDF storage
// ---------------------------------------------------------------------------

let db = null;
const DB_NAME = 'RezumeTailoredPdfs';
const DB_VERSION = 1;
const STORE_NAME = 'pdfs';

/**
 * Initialize IndexedDB for storing tailored PDFs
 */
async function initIndexedDB() {
  return new Promise((resolve, reject) => {
    const request = indexedDB.open(DB_NAME, DB_VERSION);

    request.onerror = () => reject(request.error);
    request.onsuccess = () => {
      db = request.result;
      resolve(db);
    };

    request.onupgradeneeded = (event) => {
      const database = event.target.result;
      if (!database.objectStoreNames.contains(STORE_NAME)) {
        database.createObjectStore(STORE_NAME, { keyPath: 'jobUrl' });
      }
    };
  });
}

/**
 * Save tailored text to IndexedDB (not the full PDF - we can regenerate it)
 */
async function saveTailoredTextToDB(jobUrl, tailoredText, filename) {
  if (!db) await initIndexedDB();

  return new Promise((resolve, reject) => {
    const transaction = db.transaction([STORE_NAME], 'readwrite');
    const store = transaction.objectStore(STORE_NAME);

    const data = {
      jobUrl: jobUrl,
      tailoredText: tailoredText, // Store only text (5-15KB) instead of PDF blob (500KB-2MB)
      filename: filename,
      timestamp: Date.now()
    };

    const request = store.put(data);
    request.onsuccess = () => resolve();
    request.onerror = () => reject(request.error);
  });
}

/**
 * Get tailored text from IndexedDB
 */
async function getTailoredTextFromDB(jobUrl) {
  if (!db) await initIndexedDB();

  return new Promise((resolve, reject) => {
    const transaction = db.transaction([STORE_NAME], 'readonly');
    const store = transaction.objectStore(STORE_NAME);
    const request = store.get(jobUrl);

    request.onsuccess = () => resolve(request.result || null);
    request.onerror = () => reject(request.error);
  });
}

/**
 * Delete tailored text from IndexedDB
 */
async function deleteTailoredTextFromDB(jobUrl) {
  if (!db) await initIndexedDB();

  return new Promise((resolve, reject) => {
    const transaction = db.transaction([STORE_NAME], 'readwrite');
    const store = transaction.objectStore(STORE_NAME);
    const request = store.delete(jobUrl);

    request.onsuccess = () => resolve();
    request.onerror = () => reject(request.error);
  });
}

/**
 * Clear all tailored texts from IndexedDB
 */
async function clearAllTailoredTexts() {
  if (!db) await initIndexedDB();

  return new Promise((resolve, reject) => {
    const transaction = db.transaction([STORE_NAME], 'readwrite');
    const store = transaction.objectStore(STORE_NAME);
    const request = store.clear();

    request.onsuccess = () => resolve();
    request.onerror = () => reject(request.error);
  });
}

// ---------------------------------------------------------------------------
// Resume keyword extraction (client-side, using PDF.js)
// ---------------------------------------------------------------------------

/**
 * Extracts plain text from a base64-encoded PDF using the bundled PDF.js.
 */
async function extractTextFromPDF(base64) {
  const binary = atob(base64);
  const bytes  = new Uint8Array(binary.length);
  for (let i = 0; i < binary.length; i++) bytes[i] = binary.charCodeAt(i);

  const loadingTask = pdfjsLib.getDocument({ data: bytes });
  const pdf         = await loadingTask.promise;

  let text = '';
  for (let pageNum = 1; pageNum <= pdf.numPages; pageNum++) {
    const page    = await pdf.getPage(pageNum);
    const content = await page.getTextContent();
    text += content.items.map(item => item.str).join(' ') + '\n';
  }
  return text;
}

/**
 * Returns the subset of SWE_TECH_KEYWORDS that appear in the given text.
 */
function extractKeywordsFromText(text) {
  const lower = text.toLowerCase();
  return SWE_TECH_KEYWORDS.filter(kw => lower.includes(kw));
}

/**
 * Kicks off async resume keyword extraction from the cached base64 PDF.
 */
async function extractResumeKeywords(base64) {
  try {
    const text = await extractTextFromPDF(base64);
    cachedResumeKeywords = extractKeywordsFromText(text);
  } catch (_) {
    cachedResumeKeywords = [];
  }
}

// ---------------------------------------------------------------------------
// Score ring animation
// ---------------------------------------------------------------------------
function animateScore(score) {
  const circumference = 188.5; // 2 * PI * 30
  const offset = circumference - (score / 100) * circumference;

  // Color by score
  let color = '#dc2626'; // red
  if (score >= 75) color = '#059669';       // green
  else if (score >= 55) color = '#2563eb';  // blue
  else if (score >= 35) color = '#d97706';  // amber

  scoreRingFill.style.stroke = color;
  scoreRingFill.style.strokeDashoffset = offset;
  scoreValue.textContent = score;
}

function verdictFromScore(score) {
  if (score >= 75) return { label: 'Strong match',  cls: 'verdict-strong' };
  if (score >= 55) return { label: 'Good match',    cls: 'verdict-good'   };
  if (score >= 35) return { label: 'Fair match',    cls: 'verdict-fair'   };
  return             { label: 'Weak match',    cls: 'verdict-weak'   };
}

function renderPills(container, keywords, pillClass) {
  container.innerHTML = '';
  keywords.forEach(kw => {
    const span = document.createElement('span');
    span.className = `pill ${pillClass}`;
    span.textContent = kw;
    container.appendChild(span);
  });
}

/**
 * Shows a celebration toast when match score improves after tailoring
 */
function showCelebration(oldScore, newScore) {
  const improvement = newScore - oldScore;
  if (improvement <= 0) return; // Only celebrate improvements

  // Create celebration toast
  const toast = document.createElement('div');
  toast.className = 'celebration-toast';
  toast.innerHTML = `
    <span class="celebration-emoji">🎉</span>
    <div>Score Improved!</div>
    <div class="celebration-text">+${improvement} points (${oldScore} → ${newScore})</div>
  `;

  document.body.appendChild(toast);

  // Add pulse animation to score ring
  const scoreRing = document.querySelector('.score-ring');
  scoreRing.classList.add('celebrate');

  // Remove toast and animation after 2 seconds
  setTimeout(() => {
    toast.remove();
    scoreRing.classList.remove('celebrate');
  }, 2000);
}

// ---------------------------------------------------------------------------
// View switchers
// ---------------------------------------------------------------------------
function showLoginView() {
  hide(viewMain);
  hide(userChip);
  hide(btnLogout);
  show(viewLogin);
  clearStatus();
}

function showMainView(user) {
  hide(viewLogin);
  show(viewMain);
  show(userChip);
  show(btnLogout);

  // Populate user chip
  userAvatar.textContent = initials(user.name);
  userNameEl.textContent = user.name || 'Anonymous';
  userEmailEl.textContent = user.email || '';

  // Trigger JD extraction from current tab
  extractJDFromTab();
}

// ---------------------------------------------------------------------------
// Auth flow
// ---------------------------------------------------------------------------
async function loadAuthState() {
  const data = await bg('getAuth');
  cachedToken = data.token;
  cachedUser  = data.user;

  if (cachedToken && cachedUser) {
    // Load cached resume
    const resumeData = await bg('getResume');
    cachedResumeBase64 = resumeData.resumeBase64;
    cachedResumeName   = resumeData.resumeName;

    // Kick off background keyword extraction — non-blocking
    if (cachedResumeBase64) extractResumeKeywords(cachedResumeBase64);

    showMainView(cachedUser);
  } else {
    showLoginView();
  }
}

async function doLogin() {
  if (!selectedLoginFile) return;

  setStatus('Logging in...', 'loading');
  btnLogin.disabled = true;

  try {
    const formData = new FormData();
    formData.append('pdf', selectedLoginFile);

    // Include Authorization header if we have an existing token
    const headers = {};
    if (cachedToken) {
      headers['Authorization'] = `Bearer ${cachedToken}`;
    }

    const resp = await fetch(`${API_BASE}/api/auth/resume-login`, {
      method: 'POST',
      headers,
      body: formData,
    });

    const json = await resp.json();

    if (!json.success) {
      setStatus(json.error || 'Login failed. Check the server is running.', 'error');
      btnLogin.disabled = false;
      return;
    }

    // Store token + user
    cachedToken = json.token;
    cachedUser  = json.user;
    await bg('saveAuth', { token: json.token, user: json.user });

    // Cache resume as base64 for future matching
    cachedResumeBase64 = await fileToBase64(selectedLoginFile);
    cachedResumeName   = selectedLoginFile.name;
    await bg('saveResume', { resumeBase64: cachedResumeBase64, resumeName: cachedResumeName });

    // Kick off background keyword extraction — non-blocking
    extractResumeKeywords(cachedResumeBase64);

    clearStatus();
    showMainView(cachedUser);

  } catch (err) {
    console.error('[Rezume] Login error:', err);
    setStatus('Network error: Cannot reach server.', 'error');
    btnLogin.disabled = false;
  }
}

async function doLogout() {
  await bg('logout');
  cachedToken = cachedUser = cachedResumeBase64 = cachedResumeName = null;
  cachedResumeKeywords = null;
  selectedLoginFile = null;
  extractedJD = null;
  previousMatchScore = null;
  tailoredText = tailoredPdfBlob = null;
  matchCache = {}; // Clear all cached results

  // Clear IndexedDB
  try {
    await clearAllTailoredTexts();
    console.log('[Rezume] Cleared all tailored texts from IndexedDB');
  } catch (err) {
    console.error('[Rezume] Failed to clear IndexedDB:', err);
  }

  hide(scoreCard);
  hide(keywordsCard);
  hide(tailorCTACard);
  showLoginView();
}

// ---------------------------------------------------------------------------
// JD extraction
// ---------------------------------------------------------------------------

// URL schemes where content scripts cannot be injected
const RESTRICTED_PREFIXES = [
  'chrome://', 'chrome-extension://', 'edge://', 'about:', 'data:',
];

function isRestrictedUrl(url) {
  if (!url) return true;
  return RESTRICTED_PREFIXES.some(p => url.startsWith(p));
}

async function extractJDFromTab() {
  jdMeta.textContent = 'Extracting job description from page...';
  hide(jdPreview);
  hide(scoreCard);
  hide(keywordsCard);
  hide(tailorCTACard);
  previousMatchScore = null; // Reset for new JD
  clearStatus();

  console.log('[Rezume] extractJDFromTab() called');

  try {
    // In a side panel, currentWindow may not always resolve — fall back to lastFocusedWindow
    let [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    console.log('[Rezume] currentWindow query result:', tab ? `id=${tab.id} url=${tab.url}` : 'null');

    if (!tab) {
      [tab] = await chrome.tabs.query({ active: true, lastFocusedWindow: true });
      console.log('[Rezume] lastFocusedWindow fallback result:', tab ? `id=${tab.id} url=${tab.url}` : 'null');
    }

    if (!tab) {
      console.warn('[Rezume] No active tab found after both queries');
      jdMeta.textContent = 'No active tab found.';
      return;
    }

    // Browser-internal pages (chrome://, about:newtab, etc.) cannot run content scripts
    if (isRestrictedUrl(tab.url)) {
      console.log('[Rezume] Restricted URL — skipping injection:', tab.url);
      jdMeta.textContent = 'Open a job listing page to get started.';
      return;
    }

    console.log('[Rezume] Sending extractJD message to tab', tab.id, tab.url);

    // Try sending to an already-loaded content script first
    let response;
    try {
      response = await chrome.tabs.sendMessage(tab.id, { action: 'extractJD' });
      console.log('[Rezume] Got response from existing content script:', response?.success, 'method:', response?.method, 'textLen:', response?.text?.length);
    } catch (sendErr) {
      console.warn('[Rezume] sendMessage failed (no content script yet):', sendErr.message);
      // No listener — inject the content script manually, then retry
      try {
        console.log('[Rezume] Injecting content.js...');
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          files: ['content.js'],
        });
        console.log('[Rezume] Injection succeeded, retrying sendMessage...');
        response = await chrome.tabs.sendMessage(tab.id, { action: 'extractJD' });
        console.log('[Rezume] Got response after injection:', response?.success, 'method:', response?.method, 'textLen:', response?.text?.length);
      } catch (injectErr) {
        // Page still not injectable (e.g. PDF viewer, local file without permission)
        console.error('[Rezume] Injection also failed:', injectErr.message);
        jdMeta.textContent = 'Could not access this page. Try a job board like LinkedIn or Indeed.';
        return;
      }
    }

    if (response && response.success && response.text && response.text.length > 100) {
      console.log('[Rezume] JD extracted successfully. confidence:', response.confidence, 'chars:', response.text.length);
      extractedJD = response;

      const preview = response.text.substring(0, 300).replace(/\n+/g, ' ').trim();
      const confidenceLabel = response.confidence === 'high'   ? '✓'
                            : response.confidence === 'medium' ? '~'
                            : '';
      jdMeta.textContent = `${confidenceLabel} Detected on: ${response.title || tab.url}`.trim();
      jdPreview.textContent = preview + (response.text.length > 300 ? '...' : '');
      show(jdPreview);

      // Check if we have cached results for this job URL
      const jobUrl = response.url || tab.url;
      const cached = matchCache[jobUrl];

      if (cached && cached.matchResults) {
        console.log('[Rezume] Found cached match results for:', jobUrl);

        // Show loading state for minimum 300ms (same UX as API call)
        const startTime = Date.now();
        setStatus('Evaluating...', 'loading');
        hide(scoreCard);
        hide(keywordsCard);
        hide(tailorCTACard);
        hide(tailorCard);

        // Restore from cache
        lastMatchResults = {
          extractedText: cached.extractedText,
          matchData: cached.matchData,
          analysisData: cached.analysisData
        };
        previousMatchScore = cached.previousScore;
        tailoredPdfBlob = cached.tailoredPdfBlob || null;
        tailoredText = cached.tailoredText || null;

        // Ensure loading shows for at least 300ms
        const elapsed = Date.now() - startTime;
        const remaining = Math.max(0, 300 - elapsed);

        setTimeout(() => {
          clearStatus();
          // Render cached results
          renderResults(cached.matchResults);

          // Restore tailored PDF UI if exists in memory cache
          if (cached.tailoredPdfBlob && cached.tailoredFilename) {
            show(tailorCard);
            show(tailorDone);
            hide(tailorProgress);
            hide(tailorCTACard);
            tailorDoneLabel.textContent = cached.tailoredFilename;
            btnDownloadTailor.dataset.filename = cached.tailoredFilename;
          } else {
            // Not in memory cache - try loading from IndexedDB
            getTailoredTextFromDB(jobUrl).then(stored => {
              if (stored && stored.tailoredText) {
                console.log('[Rezume] Found tailored text in IndexedDB for:', jobUrl);
                tailoredText = stored.tailoredText;
                show(tailorCard);
                show(tailorDone);
                hide(tailorProgress);
                hide(tailorCTACard);
                tailorDoneLabel.textContent = stored.filename;
                btnDownloadTailor.dataset.filename = stored.filename;
              }
            }).catch(err => {
              console.error('[Rezume] Failed to load from IndexedDB:', err);
            });
          }
        }, remaining);
      } else {
        // No cache - run the full server-side match
        doMatch();
      }
    } else {
      console.warn('[Rezume] Response did not contain usable JD text:', response);
      extractedJD = null;
      jdMeta.textContent = 'No job description found on this page.';
    }
  } catch (err) {
    console.error('[Rezume] Unexpected error in extractJDFromTab:', err.message, err.stack);
    extractedJD = null;
    jdMeta.textContent = 'Could not read this page. Try navigating to a job listing.';
  }
}

// ---------------------------------------------------------------------------
// Match resume to JD
// ---------------------------------------------------------------------------
async function doMatch() {
  if (!extractedJD) return;

  const startTime = Date.now();
  setStatus('Evaluating...', 'loading');
  hide(scoreCard);
  hide(keywordsCard);
  hide(tailorCTACard);
  hide(tailorCard);

  try {
    if (!cachedResumeBase64) {
      setStatus('No resume available. Please login with your resume.', 'error');
      return;
    }

    const resumeBlob = base64ToBlob(cachedResumeBase64);
    const resumeName = cachedResumeName || 'resume.pdf';

    const formData = new FormData();
    formData.append('pdf', resumeBlob, resumeName);
    formData.append('jobDescription', extractedJD.text);

    const headers = {};
    if (cachedToken) headers['Authorization'] = `Bearer ${cachedToken}`;

    const resp = await fetch(`${API_BASE}/api/match`, {
      method: 'POST',
      headers,
      body: formData,
    });

    if (!resp.ok) {
      const errorText = await resp.text().catch(() => 'Unknown error');
      setStatus(`Server error (${resp.status}): ${errorText}`, 'error');
      return;
    }

    const json = await resp.json();

    if (!json.success) {
      setStatus(json.error || 'Match failed.', 'error');
      return;
    }

    // Ensure loading shows for at least 300ms
    const elapsed = Date.now() - startTime;
    const remaining = Math.max(0, 300 - elapsed);

    if (remaining > 0) {
      await new Promise(resolve => setTimeout(resolve, remaining));
    }

    clearStatus();
    renderResults(json.data);

  } catch (err) {
    console.error('[Rezume] Match error:', err);
    // Network error or JSON parsing error
    if (err.name === 'TypeError' && err.message.includes('fetch')) {
      setStatus('Network error: Cannot reach server.', 'error');
    } else {
      setStatus(`Error: ${err.message}`, 'error');
    }
  }
}

// ---------------------------------------------------------------------------
// Results rendering
// ---------------------------------------------------------------------------
function renderResults(data, fromTailoredPdf = false) {
  try {
    if (!data) {
      throw new Error('No data received from server');
    }

    const analysis = data.analysis || {};
    const match    = data.match    || {};

    // Clear previous tailored resume UI (new match means old tailored resume is outdated)
    // BUT: Don't clear if this update is FROM the tailored PDF itself
    if (!fromTailoredPdf) {
      hide(tailorCard);
      tailoredPdfBlob = null;
    }

    // Primary score: how well the resume matches THIS specific JD
    const matchScore = match.matchScore ?? 0;
    const atsScore   = analysis.atsScore ?? 0;

    // Show celebration if this is from tailored PDF and score improved
    if (fromTailoredPdf && previousMatchScore !== null && matchScore > previousMatchScore) {
      showCelebration(previousMatchScore, matchScore);
    }

    // Store current score for future comparison
    if (!fromTailoredPdf) {
      previousMatchScore = matchScore;
    }

    animateScore(matchScore);

    // Level + meta line
    const levelLabel = (analysis.level || 'unknown').replace(/\b\w/g, c => c.toUpperCase());
    scoreLevel.textContent = levelLabel;

    const totalJDKw    = match.summary?.totalJDKeywords ?? 0;
    const totalMatched = match.summary?.totalMatched     ?? 0;
    scoreMeta.textContent = totalJDKw > 0
      ? `${totalMatched} of ${totalJDKw} JD keywords · ATS score: ${atsScore}`
      : `${analysis.keywordCount ?? 0} keywords found · ATS score: ${atsScore}`;

    const verdict = verdictFromScore(matchScore);
    scoreVerdict.className   = `score-verdict ${verdict.cls}`;
    scoreVerdict.textContent = verdict.label;

    // JD-specific missing keywords — what this job wants that the resume doesn't have
    const jdMissing = [...new Set(Object.values(match.missing?.byCategory || {}).flat())];
    if (jdMissing.length > 0) {
      renderPills(criticalPills, jdMissing.slice(0, 10), 'pill-critical');
      show(criticalGroup);
    } else {
      hide(criticalGroup);
    }

    // General ATS improvements — resume-level suggestions independent of this JD
    const atsImprovements = [
      ...(analysis.suggestions?.critical    || []),
      ...(analysis.suggestions?.recommended || []),
    ];
    if (atsImprovements.length > 0) {
      renderPills(recommendedPills, atsImprovements.slice(0, 8), 'pill-recommended');
      show(recommendedGroup);
    } else {
      hide(recommendedGroup);
    }

    // Keywords that ARE in both the resume and this JD
    const matchedInJD = [...new Set(Object.values(match.matched?.sample || {}).flat())];
    const foundKeywords = matchedInJD.length > 0
      ? matchedInJD
      : Object.values(analysis.foundKeywords || {}).flat().slice(0, 10);

    if (foundKeywords.length > 0) {
      renderPills(foundPills, [...new Set(foundKeywords)].slice(0, 10), 'pill-found');
      show(foundGroup);
    } else {
      hide(foundGroup);
    }

    // Cache match results for JD tailoring feature (full structured data for better AI optimization)
    lastMatchResults = {
      extractedText: data.extractedText || '',
      matchData: match,  // Full match object with matched/missing/extra keywords by category
      analysisData: analysis  // Full analysis object with suggestions and structure
    };

    console.log('[Rezume] Cached match results for tailoring:', {
      textLength: lastMatchResults.extractedText.length,
      totalMissing: match.missing?.total || 0,
      criticalMissing: match.missing?.categorized?.critical?.length || 0,
      importantMissing: match.missing?.categorized?.important?.length || 0,
      matchedCount: match.matched?.count || 0
    });

    // Save to URL-based cache (only if not from tailored PDF update)
    if (!fromTailoredPdf && extractedJD && extractedJD.url) {
      const jobUrl = extractedJD.url;

      // Limit cache size to last 10 jobs
      const cacheKeys = Object.keys(matchCache);
      if (cacheKeys.length >= 10) {
        // Remove oldest entry
        const oldestKey = cacheKeys.reduce((oldest, key) =>
          !oldest || matchCache[key].timestamp < matchCache[oldest].timestamp ? key : oldest
        );
        delete matchCache[oldestKey];
        console.log('[Rezume] Removed oldest cache entry:', oldestKey);
      }

      matchCache[jobUrl] = {
        matchResults: data,
        extractedText: lastMatchResults.extractedText,
        matchData: lastMatchResults.matchData,
        analysisData: lastMatchResults.analysisData,
        tailoredPdfBlob: null, // Will be set when tailoring completes
        tailoredFilename: null,
        previousScore: matchScore,
        timestamp: Date.now()
      };

      console.log('[Rezume] Saved match results to cache for:', jobUrl);
    }

    show(scoreCard);
    show(keywordsCard);
    show(tailorCTACard);

  } catch (err) {
    console.error('[Rezume] Error rendering results:', err);
    setStatus(`Failed to render results: ${err.message}`, 'error');
  }
}

// ---------------------------------------------------------------------------
// PDF rendering helpers (ported from app.js, adapted for Blob input + 2× scale)
// ---------------------------------------------------------------------------

/**
 * Renders each page of a PDF blob to a PNG Blob using PDF.js.
 * Uses scale 2.0 (≈144 DPI) — good quality without excessive memory.
 * @param {Blob} pdfBlob
 * @param {number} [scale=2.0]
 * @returns {Promise<Blob[]>}
 */
async function renderPDFToImage(pdfBlob, scale = 2.0) {
  const arrayBuffer = await pdfBlob.arrayBuffer();
  const loadingTask = pdfjsLib.getDocument({ data: arrayBuffer });
  const pdf         = await loadingTask.promise;

  const imageBlobs = [];
  for (let pageNum = 1; pageNum <= pdf.numPages; pageNum++) {
    const page     = await pdf.getPage(pageNum);
    const viewport = page.getViewport({ scale });
    const canvas   = document.createElement('canvas');
    const ctx      = canvas.getContext('2d', { alpha: false });
    canvas.width   = viewport.width;
    canvas.height  = viewport.height;

    ctx.imageSmoothingEnabled  = true;
    ctx.imageSmoothingQuality  = 'high';

    await page.render({
      canvasContext: ctx,
      viewport,
      intent: 'print',
      renderInteractiveForms: false,
    }).promise;

    const blob = await new Promise((resolve, reject) => {
      canvas.toBlob(b => b ? resolve(b) : reject(new Error('Canvas to blob failed')), 'image/png', 1.0);
    });
    imageBlobs.push(blob);
  }
  return imageBlobs;
}

/**
 * Creates a PDF with a hidden text layer (ATS) and visible image pages.
 * @param {string}  text        - The tailored text (hidden layer)
 * @param {Blob[]}  imageBlobs  - Page images, one per page
 * @param {number}  [renderScale=2.0] - Scale used when rendering images (pixels-per-point)
 * @returns {Promise<Blob>}
 */
async function createEnhancedPDF(text, imageBlobs, renderScale = 2.0) {
  const { PDFDocument, rgb } = PDFLib;
  const pdfDoc = await PDFDocument.create();

  const pageWidth  = 612;  // US Letter: 8.5" × 11"
  const pageHeight = 792;
  const fontSize   = 10;
  const lineHeight = 12;
  const margin     = 50;

  const allLines           = text.split('\n');
  const totalPages         = imageBlobs.length;
  const linesPerPage       = Math.ceil(allLines.length / totalPages);

  for (let i = 0; i < totalPages; i++) {
    const page          = pdfDoc.addPage([pageWidth, pageHeight]);
    const imageAB       = await imageBlobs[i].arrayBuffer();
    const pngImage      = await pdfDoc.embedPng(imageAB);
    const imageDims     = pngImage.scale(1);

    // Convert pixel dimensions to PDF points using the render scale
    const imgWPts = imageDims.width  / renderScale;
    const imgHPts = imageDims.height / renderScale;
    const fitScale = Math.min(pageWidth / imgWPts, pageHeight / imgHPts);
    const finalW   = imgWPts * fitScale;
    const finalH   = imgHPts * fitScale;
    const x        = (pageWidth  - finalW) / 2;
    const y        = (pageHeight - finalH) / 2;

    // Hidden text layer
    const startLine = i * linesPerPage;
    const pageLines = allLines.slice(startLine, startLine + linesPerPage);
    let   curY      = pageHeight - margin;

    for (const line of pageLines) {
      if (line.trim()) {
        page.drawText(line.substring(0, 150), {
          x: margin, y: curY, size: fontSize, color: rgb(1, 1, 1),
        });
      }
      curY -= lineHeight;
      if (curY < margin) break;
    }

    // White cover + image overlay
    page.drawRectangle({ x: 0, y: 0, width: pageWidth, height: pageHeight, color: rgb(1, 1, 1) });
    page.drawImage(pngImage, { x, y, width: finalW, height: finalH });
  }

  const pdfBytes = await pdfDoc.save();
  return new Blob([pdfBytes], { type: 'application/pdf' });
}

// ---------------------------------------------------------------------------
// JD Tailoring
// ---------------------------------------------------------------------------

/**
 * Orchestrates the full tailor flow:
 *   1. POST resume + JD to /api/jd-tailor → tailoredText
 *   2. renderPDFToImage on the cached resume blob
 *   3. createEnhancedPDF(tailoredText, pageBlobs)
 *   4. Store blob + reveal download card
 */
async function doTailor() {
  if (!extractedJD) return;

  if (!lastMatchResults || !lastMatchResults.extractedText) {
    setStatus('Please run a match analysis first before tailoring.', 'error');
    return;
  }

  if (!cachedResumeBase64) {
    setStatus('No resume available. Please login with your resume.', 'error');
    return;
  }

  const resumeBlob = base64ToBlob(cachedResumeBase64);
  const resumeName = cachedResumeName || 'resume.pdf';

  // UI: show tailor card + progress, hide CTA button
  show(tailorCard);
  show(tailorProgress);
  hide(tailorDone);
  hide(tailorCTACard);

  const totalMissing = lastMatchResults.matchData?.missing?.total || 0;
  const criticalMissing = lastMatchResults.matchData?.missing?.categorized?.critical?.length || 0;

  tailorProgressText.textContent = `Step 1/4: Analyzing ${totalMissing} missing keywords (${criticalMissing} critical)...`;
  btnTailor.disabled = true;

  console.log('[Rezume] Starting tailoring with:', {
    totalMissing: totalMissing,
    criticalMissing: criticalMissing,
    matchedCount: lastMatchResults.matchData?.matched?.count || 0,
    textLength: lastMatchResults.extractedText.length
  });

  try {
    // Step 1: Get tailored text from server (send JSON, not FormData)
    tailorProgressText.textContent = `Step 1/4: Sending to AI for tailoring (${totalMissing} keywords, ${criticalMissing} critical)...`;

    const headers = {
      'Content-Type': 'application/json'
    };
    if (cachedToken) headers['Authorization'] = `Bearer ${cachedToken}`;

    const resp = await fetch(`${API_BASE}/api/jd-tailor`, {
      method: 'POST',
      headers,
      body: JSON.stringify({
        resumeText: lastMatchResults.extractedText,
        matchData: lastMatchResults.matchData,
        analysisData: lastMatchResults.analysisData
      }),
    });

    if (!resp.ok) {
      const errorText = await resp.text().catch(() => 'Unknown error');
      throw new Error(`Server error (${resp.status}): ${errorText}`);
    }

    const json = await resp.json();
    if (!json.success) {
      throw new Error(json.error || 'Tailoring failed');
    }

    // Store tailored text for future regeneration
    tailoredText = json.tailoredText;
    console.log('[Rezume] Received tailored text:', tailoredText.length, 'chars');

    // Step 2: Render PDF pages to images
    tailorProgressText.textContent = 'Step 2/4: Rendering resume pages to high-quality images...';
    const pageBlobs = await renderPDFToImage(resumeBlob, 2.0);

    // Step 3: Build the tailored PDF
    tailorProgressText.textContent = 'Step 3/4: Building your tailored PDF with ATS optimization...';
    tailoredPdfBlob = await createEnhancedPDF(tailoredText, pageBlobs, 2.0);

    // Prepare filename for download card
    const baseName = resumeName.replace(/\.pdf$/i, '');
    const tailoredFilename = `tailored-${baseName}.pdf`;

    // Step 4: Re-run match analysis with tailored PDF to show improvements
    tailorProgressText.textContent = 'Step 4/4: Analyzing improvements...';

    const matchFormData = new FormData();
    matchFormData.append('pdf', tailoredPdfBlob, resumeName);
    matchFormData.append('jobDescription', extractedJD.text);

    const matchHeaders = {};
    if (cachedToken) matchHeaders['Authorization'] = `Bearer ${cachedToken}`;

    const matchResp = await fetch(`${API_BASE}/api/match`, {
      method: 'POST',
      headers: matchHeaders,
      body: matchFormData,
    });

    if (matchResp.ok) {
      const matchJson = await matchResp.json();
      if (matchJson.success) {
        // Update UI with improved match scores (temporary - doesn't change cached resume)
        // Pass true to preserve the tailored PDF download section
        renderResults(matchJson.data, true);

        // Update cache with improved match results
        if (extractedJD && extractedJD.url && matchCache[extractedJD.url]) {
          matchCache[extractedJD.url].matchResults = matchJson.data;
          console.log('[Rezume] Updated cache with tailored match results');
        }

        console.log('[Rezume] Match results updated with tailored PDF');
      }
    } else {
      console.warn('[Rezume] Failed to re-run match analysis:', matchResp.status);
    }

    // Show download card
    hide(tailorProgress);
    tailorDoneLabel.textContent  = tailoredFilename;
    btnDownloadTailor.dataset.filename = tailoredFilename;
    show(tailorDone);

    // Save tailored text to IndexedDB and memory cache for this job URL
    if (extractedJD && extractedJD.url) {
      const jobUrl = extractedJD.url;

      // Save to IndexedDB (persistent storage - survives extension reload)
      try {
        await saveTailoredTextToDB(jobUrl, tailoredText, tailoredFilename);
        console.log('[Rezume] Saved tailored text to IndexedDB for:', jobUrl);
      } catch (err) {
        console.error('[Rezume] Failed to save to IndexedDB:', err);
      }

      // Save to memory cache (fast access during current session)
      if (matchCache[jobUrl]) {
        matchCache[jobUrl].tailoredText = tailoredText;
        matchCache[jobUrl].tailoredFilename = tailoredFilename;
        matchCache[jobUrl].tailoredPdfBlob = tailoredPdfBlob; // Keep blob in memory for current session
        console.log('[Rezume] Saved tailored text to memory cache for:', jobUrl);
      }
    }

    console.log('[Rezume] Tailoring complete!');
    tailorCard.scrollIntoView({ behavior: 'smooth', block: 'nearest' });

  } catch (err) {
    console.error('[Rezume] Tailoring error:', err);
    // Network error detection
    if (err.name === 'TypeError' && err.message.includes('fetch')) {
      setStatus('Network error: Cannot reach server.', 'error');
    } else {
      setStatus(`Tailoring failed: ${err.message}`, 'error');
    }
    hide(tailorProgress);
  } finally {
    btnTailor.disabled = false;
  }
}

async function downloadTailored() {
  const filename = btnDownloadTailor.dataset.filename || 'tailored-resume.pdf';

  // If we have the PDF blob in memory, download it directly
  if (tailoredPdfBlob) {
    const url = URL.createObjectURL(tailoredPdfBlob);
    const a   = document.createElement('a');
    a.href     = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    URL.revokeObjectURL(url);
    a.remove();
    console.log('[Rezume] Downloaded from memory:', filename);
    return;
  }

  // PDF not in memory - try to regenerate from stored text
  if (!tailoredText && extractedJD && extractedJD.url) {
    // Try loading from IndexedDB
    try {
      const stored = await getTailoredTextFromDB(extractedJD.url);
      if (stored && stored.tailoredText) {
        tailoredText = stored.tailoredText;
        console.log('[Rezume] Loaded tailored text from IndexedDB');
      }
    } catch (err) {
      console.error('[Rezume] Failed to load from IndexedDB:', err);
    }
  }

  if (!tailoredText) {
    setStatus('Tailored resume not available. Please tailor again.', 'error');
    return;
  }

  // Regenerate the PDF from stored text
  try {
    setStatus('Preparing download...', 'loading');

    // Render original resume pages to images
    const resumeBlob = base64ToBlob(cachedResumeBase64);
    const pageBlobs = await renderPDFToImage(resumeBlob, 2.0);

    // Create PDF with tailored text
    tailoredPdfBlob = await createEnhancedPDF(tailoredText, pageBlobs, 2.0);

    // Download the regenerated PDF
    const url = URL.createObjectURL(tailoredPdfBlob);
    const a   = document.createElement('a');
    a.href     = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    URL.revokeObjectURL(url);
    a.remove();

    clearStatus();
    console.log('[Rezume] Downloaded (regenerated from stored text):', filename);
  } catch (err) {
    console.error('[Rezume] Failed to regenerate PDF:', err);
    setStatus(`Failed to download: ${err.message}`, 'error');
  }
}

// ---------------------------------------------------------------------------
// Event listeners
// ---------------------------------------------------------------------------

// Drop zone click → trigger file input
dropZone.addEventListener('click', () => resumeFileInput.click());

// Drag and drop
dropZone.addEventListener('dragover', (e) => {
  e.preventDefault();
  dropZone.classList.add('drag-over');
});
dropZone.addEventListener('dragleave', () => dropZone.classList.remove('drag-over'));
dropZone.addEventListener('drop', (e) => {
  e.preventDefault();
  dropZone.classList.remove('drag-over');
  const file = e.dataTransfer.files[0];
  if (file && file.type === 'application/pdf') {
    onLoginFileSelected(file);
  }
});

resumeFileInput.addEventListener('change', () => {
  const file = resumeFileInput.files[0];
  if (file) onLoginFileSelected(file);
});

function onLoginFileSelected(file) {
  selectedLoginFile = file;
  loginFileName.textContent = file.name;
  show(loginFileName);
  btnLogin.disabled = false;

  // Auto-login when file is selected
  doLogin();
}

btnLogin.addEventListener('click', doLogin);
btnLogout.addEventListener('click', doLogout);

btnChangeResume.addEventListener('click', async () => {
  // Clear cached resume, return to login so user can upload a new one
  await bg('clearResume');
  cachedResumeBase64 = cachedResumeName = null;
  selectedLoginFile  = null;
  tailoredText = tailoredPdfBlob = null;
  matchCache = {}; // Clear all cached match results (tied to previous resume)

  // Clear IndexedDB (tailored resumes are tied to the original resume)
  try {
    await clearAllTailoredTexts();
    console.log('[Rezume] Cleared all tailored texts from IndexedDB');
  } catch (err) {
    console.error('[Rezume] Failed to clear IndexedDB:', err);
  }

  // Reset login form
  loginFileName.textContent = '';
  hide(loginFileName);
  btnLogin.disabled = true;
  showLoginView();
});

btnTailor.addEventListener('click', doTailor);
btnDownloadTailor.addEventListener('click', downloadTailored);

// Make the tailored PDF card draggable to file inputs
tailorDone.addEventListener('dragstart', (e) => {
  if (!tailoredPdfBlob) {
    e.preventDefault();
    return;
  }

  const filename = btnDownloadTailor.dataset.filename || 'tailored-resume.pdf';

  // Create a File object from the blob
  const file = new File([tailoredPdfBlob], filename, { type: 'application/pdf' });

  // Add the file to the drag data
  e.dataTransfer.items.add(file);
  e.dataTransfer.effectAllowed = 'copy';

  console.log('[Rezume] Starting drag for:', filename);
});

tailorDone.addEventListener('dragend', () => {
  console.log('[Rezume] Drag ended');
});

// Prevent the download button from triggering drag
btnDownloadTailor.addEventListener('mousedown', (e) => {
  e.stopPropagation();
});

// ---------------------------------------------------------------------------
// Side panel: re-extract JD when the user switches tabs or navigates
// The panel stays alive across tab changes, so we need to watch for them.
// ---------------------------------------------------------------------------
chrome.tabs.onActivated.addListener(() => {
  if (cachedToken && cachedUser) extractJDFromTab();
});

chrome.tabs.onUpdated.addListener((tabId, changeInfo) => {
  if (changeInfo.status !== 'complete') return;
  chrome.tabs.query({ active: true, currentWindow: true }, ([activeTab]) => {
    if (activeTab && activeTab.id === tabId && cachedToken && cachedUser) {
      extractJDFromTab();
    }
  });
});

// ---------------------------------------------------------------------------
// Demo Mode
// ---------------------------------------------------------------------------

const MOCK_JD_TEXT = `About the Job
We're looking for a Senior Full-Stack Engineer to join our fast-growing team. You'll work on cutting-edge web applications using modern technologies.

What You'll Do
• Design and build scalable full-stack applications
• Work with React, Node.js, TypeScript, and PostgreSQL
• Implement CI/CD pipelines and DevOps best practices
• Collaborate with product and design teams
• Mentor junior engineers

Requirements
• 5+ years of software engineering experience
• Strong proficiency in JavaScript/TypeScript, React, Node.js
• Experience with PostgreSQL, MongoDB, or similar databases
• Familiarity with Docker, Kubernetes, AWS
• System design and architecture experience
• Strong problem-solving skills

Nice to Have
• Experience with GraphQL, Redis, Kafka
• Knowledge of microservices architecture
• Previous startup experience
• Open source contributions

We Offer
• Competitive salary and equity
• Remote-first culture
• Health insurance and 401k
• Unlimited PTO
• Professional development budget`;

const MOCK_MATCH_RESULTS = {
  success: true,
  data: {
    extractedText: "Sample resume text with keywords...",
    analysis: {
      level: 'senior',
      atsScore: 78,
      keywordCount: 42,
      foundKeywords: {
        languages: ['JavaScript', 'TypeScript', 'Python'],
        frameworks: ['React', 'Node.js', 'Express'],
        databases: ['PostgreSQL', 'MongoDB', 'Redis'],
        cloud: ['AWS', 'Docker'],
        devops: ['CI/CD', 'Kubernetes']
      },
      suggestions: {
        critical: ['System Design', 'GraphQL'],
        recommended: ['Kafka', 'Microservices', 'Redis caching']
      },
      structureRecommendations: []
    },
    match: {
      matchScore: 82,
      summary: {
        totalJDKeywords: 28,
        totalMatched: 23
      },
      matched: {
        count: 23,
        sample: {
          languages: ['JavaScript', 'TypeScript'],
          frameworks: ['React', 'Node.js'],
          databases: ['PostgreSQL', 'MongoDB'],
          cloud: ['AWS', 'Docker']
        }
      },
      missing: {
        total: 5,
        byCategory: {
          architecture: ['GraphQL', 'Microservices'],
          messaging: ['Kafka'],
          devops: ['Kubernetes advanced features']
        },
        categorized: {
          critical: ['GraphQL', 'System Design'],
          important: ['Kafka', 'Microservices']
        }
      }
    }
  }
};

async function loadDemoMode() {
  try {
    console.log('[Rezume Demo] Loading demo mode...');

    // Create mock user
    const mockUser = {
      name: 'Rajdeep Deb',
      email: 'demo@rezume.app'
    };

    // Load mock resume PDF
    const mockResumeUrl = chrome.runtime.getURL('mock-resume.pdf');
    const response = await fetch(mockResumeUrl);
    const blob = await response.blob();
    const base64 = await fileToBase64(new File([blob], 'mock-resume.pdf', { type: 'application/pdf' }));

    // Save mock auth and resume
    cachedToken = 'demo-token-' + Date.now();
    cachedUser = mockUser;
    cachedResumeBase64 = base64;
    cachedResumeName = 'mock-resume.pdf';

    await bg('saveAuth', { token: cachedToken, user: cachedUser });
    await bg('saveResume', { resumeBase64: base64, resumeName: 'mock-resume.pdf' });

    // Kick off keyword extraction
    extractResumeKeywords(base64);

    // Show main view
    showMainView(mockUser);

    // Create mock extracted JD
    extractedJD = {
      text: MOCK_JD_TEXT,
      url: 'https://example.com/jobs/senior-fullstack-engineer',
      title: 'Senior Full-Stack Engineer - Example Company',
      method: 'targeted-selector',
      confidence: 'high'
    };

    // Update JD preview
    const preview = MOCK_JD_TEXT.substring(0, 300).replace(/\n+/g, ' ').trim();
    jdMeta.textContent = '✓ Detected on: Senior Full-Stack Engineer - Example Company';
    jdPreview.textContent = preview + '...';
    show(jdPreview);

    // Show loading state
    setStatus('Evaluating...', 'loading');
    hide(scoreCard);
    hide(keywordsCard);
    hide(tailorCTACard);
    hide(tailorCard);

    // Wait minimum 300ms for better UX
    await new Promise(resolve => setTimeout(resolve, 300));

    clearStatus();

    // Render mock results
    renderResults(MOCK_MATCH_RESULTS.data);

    console.log('[Rezume Demo] Demo mode loaded successfully');

  } catch (err) {
    console.error('[Rezume Demo] Failed to load demo mode:', err);
    setStatus('Failed to load demo mode', 'error');
  }
}

// ---------------------------------------------------------------------------
// Init
// ---------------------------------------------------------------------------

// Demo button handler
document.getElementById('btnDemo')?.addEventListener('click', loadDemoMode);

loadAuthState();
